// データ処理部を独立化 (Ver.2.1以降）

class ProcessingData {
  String fullpathname;  // 読み込むCSVファイル
  String filename;      // ファイル名の部分（拡張子含む）
  
  float   [] Af;        // A相の電圧（勿論，0/1でも構わない）
  float   [] Bf;        // B相の電圧（勿論，0/1でも構わない）
  boolean [] A;         // A相の真偽
  boolean [] B;         // B相の真偽
  int     [] enc;       // ±１
  float   [] acc;       // 累積値
  float   [] pps;       // 角速度（Pulse / sec)
  float   [] rpm;       // 角速度（rpm)
  float   [] radps;     // 角速度（rad/s）
  int        max_num;   // データ数

  // コンストラクター
  ProcessingData() {
    max_num = 0;
  }
  // メモリーに読み込んだ電圧値を真偽に変換して解読（方向判別）
  void convert2boolean()  {
    // 真偽値に変換
    A = new boolean[max_num];
    B = new boolean[max_num];
    A[0] = (Af[0] >= (SH_H + SH_L) / 2.0) ? true : false;
    B[0] = (Bf[0] >= (SH_H + SH_L) / 2.0) ? true : false;
    for (int i = 1; i < max_num; i++) {
      // A相の判定
      if      ((A[i-1] == false) && (Af[i] > SH_H)) A[i] = true;
      else if ((A[i-1] == true)  && (Af[i] < SH_L)) A[i] = false;
      else                                          A[i] = A[i - 1];
      // B相の判定
      if      ((B[i-1] == false) && (Bf[i] > SH_H)) B[i] = true;
      else if ((B[i-1] == true)  && (Bf[i] < SH_L)) B[i] = false;
      else                                          B[i] = B[i - 1];
    }
  }
  boolean detectDirection() {
    // 方向判別
    int [] Ac = new int[max_num];  // A相の変化（↑:1, ↓:-1，無変化：0，[0]は使用しない）
    int [] Bc = new int[max_num];  // B相の変化（↑:1, ↓:-1，無変化：0，[0]は使用しない）
    // 変化の検出
    for (int i = 1; i < max_num; i++) {
      if (A[i] != A[i - 1]) {
        if (A[i] == true) Ac[i] =  1; 
        else              Ac[i] = -1;
      } else              Ac[i] = 0;
      if (B[i] != B[i - 1]) {
        if (B[i] == true) Bc[i] =  1; 
        else              Bc[i] = -1;
      } else              Bc[i] = 0;
    }
    enc = new int[max_num];
    enc[0] = 0;
    for (int i = 1; i < max_num; i++) {
      // チェック
      if ((Ac[i] != 0) && (Bc[i] != 0)) {
        msg.setError("判定本体で異常を検知しました．" + i + "行目のデータでA相とB相が同時に変化しています");
        return false;
      }
      if (Ac[i] != 0) {                    // A相に変化があった場合
        if (Ac[i] == 1) {                  // A相↑
          if (B[i] == false) enc[i] =  1;  // CW
          else               enc[i] = -1;  // CCW
        } else {                           // A相↓
          if (B[i] == false) enc[i] = -1;  // CCW
          else               enc[i] =  1;  // CW
        }
      } else if (Bc[i] != 0) {             // B相に変化があった場合
        if (Bc[i] == 1) {                  // B相↑
          if (A[i] == false) enc[i] = -1;  // CCW
          else               enc[i] =  1;  // CW
        } else {                           // B相↓
          if (A[i] == false) enc[i] =  1;  // CW
          else               enc[i] = -1;  // CCW
        }
      } else enc[i] = 0;
    }
    return true;
  }
  // 逆転する場合
  void reverseDirection() {
    for (int i = 0; i < max_num; i++) if (enc[i] != 0) enc[i] = -enc[i];
  }
  // カウンターの累積
  void accumlation() {
    acc = new float[max_num];
    acc[0] = 0;
    for (int i = 1; i < max_num; i++) acc[i] = acc[i - 1] + (float)enc[i];
  }
  // 角速度（pulse per sec.)
  // 参考： 2階微分フィルターによる加速度の算出法，https://tsukuba.repo.nii.ac.jp/record/6723/files/6.pdf
  //        中心差分近似, https://koki0702.github.io/dezero-book/ja/step04.html
  //                      https://qiita.com/Negelon/items/fafec9b0d33ab96d1446
  // 中心差分近似：central difference scheme ( CDS )
  void calcurateAngularVelocity() {
    pps = new float[max_num];
    int i;
    // 前方データは無効
    for (i = 0; i < CDS; i++) pps[i] = 0.0;
    // 中心差分近似処理
    for (; i < max_num - CDS; i++) {
      pps[i] = (float)((double)(acc[i + CDS] - acc[i - CDS]) / (double)(2.0 * CDS * DeltaT/1000.0));
    }
    // 後方データは無効
    for (; i < max_num; i++)  pps[i] = 0.0;
  }
  // 角速度 [rpm]
  void convert2rpm() {
    rpm = new float[max_num];
    int i;
    // 前方データは無効
    for (i = 0; i < CDS; i++) rpm[i] = 0.0;
    // 中心差分近似処理
    for (; i < max_num - CDS; i++) {
      rpm[i] = (float)((double)pps[i] * (double)60.0 / ((double)EncCount * (double)4.0));  // 四逓倍なのでロータリーエンコーダーの分解能の４倍の分解能
    }
    // 後方データは無効
    for (; i < max_num; i++)  rpm[i] = 0.0;
  }
  // 角速度 [rad/s]
  void convert2radps() {
    radps = new float[max_num];
    int i;
    // 前方データは無効
    for (i = 0; i < CDS; i++) radps[i] = 0.0;
    // 中心差分近似処理
    for (; i < max_num - CDS; i++) {
      radps[i] = (float)((double)pps[i] / ((double)EncCount * (double)4.0) * (double)TWO_PI);  // 四逓倍なのでロータリーエンコーダーの分解能の４倍の分解能
    }
    // 後方データは無効
    for (; i < max_num; i++)  radps[i] = 0.0;
  }
  // 方向転換のチェック
  // ・何回反転したか
  // ・連続して回転方向の変化し無かったサンプル数（最長，最短）
  void checkDirectionChange() {
    int direction;
    boolean ignore = true;     // データ冒頭の無変化状態を無視する
    int rev = 0;               // 状態変化の回数
    int revmin = 0, revminLine = 0;  // 連続して回転方向の変化しなかったサンプル数（最小）
    int revmax = 0, revmaxLine = 0;  // 連続して回転方向の変化しなかったサンプル数（最大）
    int plsmin = 0, plsminLine = 0;  // 連続してA/B相共にパルスが変化しなかったサンプル数（最小）
    int plsmax = 0, plsmaxLine = 0;  // 連続してA/B相共にパルスが変化しなかったサンプル数（最大）

    direction = enc[0];  // 初期状態
    revmin = revmax = 0; revminLine = revmaxLine = 1;
    plsmin = plsmax = 0; plsminLine = plsmaxLine = 1;
    int  revcnt = 0, plscnt = 0;
    for (int i = 1; i < max_num; i++) {
      // データ冒頭の無変化状態を無視する
      if (ignore) {
        if ((A[i-1] != A[i]) || (B[i-1] != B[i])) {
          ignore = false;
          direction = enc[i];
          i++;
        }
        continue;
      }
      ++revcnt;
      ++plscnt;
      // 反転回数の変化を集計
      if (enc[i] != 0) {
        if (direction != enc[i]) {  // 回転方向の変化を検出
          // 反転回数のカウント
          rev++;
          direction = enc[i];
          // 無変化期間の集計
          if ((revmin == 0) || (revcnt < revmin)) { revmin = revcnt; revminLine = i; }
          if ((revmax == 0) || (revcnt > revmax)) { revmax = revcnt; revmaxLine = i; }
          revcnt = 0;
        }
      }
      // パルス無変化の集計
      if ((A[i-1] != A[i]) || (B[i-1] != B[i])) {  // パルスの変化を検出
        // 無変化期間の集計
        if ((plsmin == 0) || (plscnt < plsmin)) { plsmin = plscnt; plsminLine = i; }
        if ((plsmax == 0) || (plscnt > plsmax)) { plsmax = plscnt; plsmaxLine = i; }
        plscnt = 0;
      }
    }
    // 結果の出力
    if (rev == 0) msg.setNotice  ("回転方向は一定方向です");
    else          msg.setWarnning("回転方向が" + rev + "回反転しました");
    msg.setNotice("回転方向無変化のサンプル数の最大，最小は，それぞれ" + revmax + "回(" + revmaxLine + "行目辺り)と" + revmin + "回(" + revminLine + "行目辺り)です");
    msg.setNotice("A/B相パルス無変化のサンプル数の最大，最小は，それぞれ" + plsmax + "回(" + plsmaxLine + "行目辺り)と" + plsmin + "回(" + plsminLine + "行目辺り)です");
  }
  // 移動平均
  // (a) パルス累積の移動平均（int)
  void movingAverageAcc() {
    // エラーチェック
    if (MApulse < 0) {
      msg.setError("移動平均（パルス累積）の項数が負なので移動平均は行いません");
      return;
    }
    if (MApulse * 2 + 1 >=  max_num) {
      msg.setError("移動平均（パルス累積）の項数がデータ数に対して大き過ぎるので移動平均は行いません");
      return;
    }
    // データ複製
    float [] acc0 = new float[max_num];
    for (int i = 0; i < max_num; i++) acc0[i] = acc[i];
    // 本体
    for (int i = MApulse; i < max_num - MApulse - 1;i++) {
      double sum = 0.0;
      for (int j = -MApulse; j <= MApulse; j++) sum += (double)acc0[i + j];
      acc[i] = (float)(sum / (double)(MApulse * 2 + 1));
    }
    msg.set("パルス累積を" + MApulse + "項で移動平均しました");
  }
  // (b) 角速度(pps)の移動平均（float)
  void movingAverageAV() {
    // エラーチェック
    if (MAav < 0) {
      msg.setError("移動平均（角速度）の項数が負なので移動平均は行いません");
      return;
    }
    if (MAav * 2 + 1 >=  max_num) {
      msg.setError("移動平均（角速度）の項数がデータ数に対して大き過ぎるので移動平均は行いません");
      return;
    }
    // データ複製
    float [] pps0 = new float[max_num];
    for (int i = 0; i < max_num; i++) pps0[i] = pps[i];
    // 本体
    for (int i = MAav + CDS; i < max_num - MAav - CDS - 1;i++) {  // CDSの分だけ前後は0のデータが入っているため（厳密にはその分はその区間で平均を取るべき）
      double sum = 0.0;
      for (int j = -MAav; j <= MAav; j++) sum += (double)pps0[i + j];
      pps[i] = (float)(sum / (double)(MAav * 2 + 1));
    }
    msg.set("角速度(pps)を" + MAav + "項で移動平均しました");
  }
  // 最小値を求める
  float minimum(float [] data) {
    float ret = data[CDS];
    for (int i = CDS + 1; i < max_num - CDS; i++) if (ret > data[i]) ret = data[i]; 
    return ret;
  }
  // 最大値を求める
  float maximum(float [] data) {
    float ret = data[CDS];
    for (int i = CDS + 1; i < max_num - CDS; i++) if (ret < data[i]) ret = data[i]; 
    return ret;
  }
  // 平均値を求める
  float mean(float [] data) {
    double ret = 0.0;
    int    n = 0;
    for (int i = CDS; i < max_num - CDS; i++, n++) ret += (double)data[i];
    ret /= (double)n;
    return (float)ret;
  }
  // 標準偏差を求める
  float sd(float [] data, float xbar) {
    double ret = 0.0;
    int    n = 0;
    for (int i = CDS; i < max_num - CDS; i++, n++) ret += (double)pow(data[i] - xbar, 2);
    ret /= (double)n;
    ret = sqrt((float)ret);
    return (float)ret;
  }
  // 角速度の元になるPulse per secを求めるCDSが適切かどうかを確認すると共に、指定されたCDSで計算を行う
  void comparePPSbyCDS() {
    int CDS0;    // オリジナルのCDSの値を一時保存
    float  max1,  max2,  max10,  max0;
    float  min1,  min2,  min10,  min0;
    float  sd1,   sd2,   sd10,   sd0;
    float  mean1, mean2, mean10, mean0;
    String str;

    CDS0 = CDS; //<>//
    // CDS = 1の場合
    CDS = 1;
    calcurateAngularVelocity();
    max1  = maximum(pps);
    min1  = minimum(pps);
    mean1 = mean(pps);
    sd1   = sd(pps, mean1);
    str   = "(参考)CDS=" + nf(CDS) + "の時のPPS（最大/最小/平均/標準偏差）：" + nf(max1) + "/" + nf(min1) + "/" + nf(mean1) + "/" + nf(sd1);
    msg.setNotice(str);
    // CDS = CDS0 * 2の場合
    CDS = CDS0 * 2;
    if (CDS * 2 < max_num) {
      calcurateAngularVelocity();
      max2  = maximum(pps);
      min2  = minimum(pps);
      mean2 = mean(pps);
      sd2  = sd(pps, mean2);
      str   = "(参考)CDS=" + nf(CDS) + "の時のPPS（最大/最小/平均/標準偏差）：" + nf(max2) + "/" + nf(min2) + "/" + nf(mean2) + "/" + nf(sd2);
      msg.setNotice(str);
    }
    // CDS = CDS0 * 10の場合
    CDS = CDS0 * 10;
    if (CDS * 2 < max_num) {
      calcurateAngularVelocity();
      max10  = maximum(pps);
      min10  = minimum(pps);
      mean10 = mean(pps);
      sd10   = sd(pps, mean10);
      str   = "(参考)CDS=" + nf(CDS) + "の時のPPS（最大/最小/平均/標準偏差）：" + nf(max10) + "/" + nf(min10) + "/" + nf(mean10) + "/" + nf(sd10);
      msg.setNotice(str);
    }
    // CDS = CDS0の場合
    CDS = CDS0;
    calcurateAngularVelocity();
    max0  = maximum(pps);
    min0  = minimum(pps);
    mean0 = mean(pps);
    sd0  = sd(pps, mean0);
    str   = "CDS=" + nf(CDS) + "の時のPPS（最大/最小/平均/標準偏差）：" + nf(max0) + "/" + nf(min0) + "/" + nf(mean0) + "/" + nf(sd0);
    msg.setNotice(str);
  }
  // 解析本体
  boolean decode(float [] Afloat, float [] Bfloat) {  
    // 読み込んだデータの正当性の再チェック
    Af = Afloat;
    Bf = Bfloat;
    if (Af.length != Bf.length) {
      msg.setError("A相のデータ数とB相のデータ数が異なります（深刻なエラー）");
      return false;
    }
    if (Af.length == 0) {
      msg.setError("読み込んだデータ数がゼロです");
      return false;
    }
    if (Af.length <= CDS * 2) {
      msg.setError("読み込んだデータ数(" + max_num + ")がCDS(" + CDS + "*2よりも少ないので角速度を計算できません");
      // 警告であって動作は終了しない
    }
    max_num = Af.length;

    // 真偽値に変換
    convert2boolean();
    // 方向判別
    if (!detectDirection()) return false;
    // 逆転する場合
    if (REVERSE) reverseDirection();
    // カウンターの累積
    accumlation();
    // 累積の移動平均
    if (MApulse > 0) movingAverageAcc();
    // 角速度 [pulse per sec.]
    // calcurateAngularVelocity();
    comparePPSbyCDS();
    // 角速度の移動平均
    if (MAav > 0) movingAverageAV();
    // 角速度 [rpm]
    convert2rpm();
    // 角速度 [rad/s]
    convert2radps();
    // 方向転換のチェック
    checkDirectionChange();
    
    return true;
  }
}

ProcessingData data = new ProcessingData(); 
